# 🚀 ADVANCED SMM PANEL - Complete Documentation

## 📋 Table of Contents
1. [System Overview](#system-overview)
2. [Key Features](#key-features)
3. [Installation Guide](#installation-guide)
4. [Configuration](#configuration)
5. [Payment Gateways](#payment-gateways)
6. [Auto-Topup System](#auto-topup-system)
7. [Architecture](#architecture)
8. [Security Features](#security-features)
9. [API Documentation](#api-documentation)
10. [Deployment](#deployment)
11. [Troubleshooting](#troubleshooting)

---

## 🎯 System Overview

**Advanced SMM Panel** is a next-generation social media marketing platform featuring:
- **Revolutionary Auto-Topup**: Automatically tops up balance during orders
- **8 Payment Gateways**: Korapay, Flutterwave, Paystack, ExpressPay, Cryptomus, Heleket, PesaPal, and more
- **Modern Stack**: PHP 8+, MySQL 8+, PDO, Argon2ID encryption
- **Enterprise Security**: Bank-grade protection with comprehensive logging
- **Beautiful UI**: Modern, animated landing page with distinctive design

---

## ✨ Key Features

### 1. **Auto-Topup System** (UNIQUE!)
- Detects insufficient balance during order placement
- Calculates exact amount needed
- Redirects to payment gateway with pre-filled amount
- After payment: automatically places order + credits remaining balance
- Session-based tracking (30-minute timeout)
- Full transaction atomicity

### 2. **Payment Integration**
- **Korapay**: Nigeria (NGN, USD)
- **Flutterwave**: Multi-country African gateway
- **Paystack**: Nigeria, Ghana, South Africa
- **ExpressPay**: Ghana payment solution
- **Cryptomus**: Cryptocurrency payments (BTC, ETH, USDT)
- **Heleket**: Ethiopian payment gateway
- **PesaPal**: East Africa (Kenya, Uganda, Tanzania)
- All with webhook support and comprehensive logging

### 3. **Security Features**
- Argon2ID password hashing
- CSRF token protection
- Rate limiting (5 attempts per 15 min)
- Session fingerprinting & hijacking detection
- SQL injection protection (PDO prepared statements)
- XSS prevention
- Secure headers (HSTS, CSP, X-Frame-Options)
- Comprehensive security logging

### 4. **Order Management**
- Standard orders
- Drip-feed campaigns
- Subscription services
- Real-time status tracking
- Automatic provider synchronization
- Refund & cancellation support

### 5. **User Features**
- Multi-role support (User, Reseller, Admin)
- API key generation
- Referral system with commissions
- Transaction history
- Activity logs
- Notification system

---

## 🛠️ Installation Guide

### Prerequisites
```bash
- PHP 8.0+
- MySQL 8.0+
- Apache/Nginx
- Composer
- SSL Certificate (required for payments)
```

### Step 1: Upload Files
```bash
# Extract the package
unzip advanced-smm-panel.zip

# Upload to your server
# Set permissions
chmod 755 -R /path/to/panel
chmod 777 -R /path/to/panel/storage
chmod 777 -R /path/to/panel/storage/logs
```

### Step 2: Database Setup
```bash
# Create database
mysql -u root -p
CREATE DATABASE smm_panel CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

# Import schema
mysql -u root -p smm_panel < database/schema.sql

# Verify installation
mysql -u root -p smm_panel -e "SHOW TABLES;"
```

### Step 3: Configuration

**config/database.php**
```php
return [
    'host' => 'localhost',
    'database' => 'smm_panel',
    'username' => 'your_username',
    'password' => 'your_password',
    'charset' => 'utf8mb4',
];
```

**config/app.php**
```php
return [
    'app_url' => 'https://yourdomain.com',
    'app_env' => 'production',
    'encryption_key' => 'GENERATE_32_RANDOM_CHARACTERS',
    // ... other settings
];
```

### Step 4: Generate Encryption Key
```bash
php -r "echo bin2hex(random_bytes(16));"
# Copy output to encryption_key in config/app.php
```

### Step 5: Test Installation
```bash
# Visit your domain
https://yourdomain.com

# Default admin login:
Username: admin
Password: Admin@123
# CHANGE IMMEDIATELY after first login!
```

---

## ⚙️ Configuration

### Payment Gateway Setup

#### Korapay Configuration
```php
// Admin Panel → Settings → Payment Methods → Korapay
{
    "public_key": "pk_test_xxxxxxxxxxxxxxxx",
    "secret_key": "sk_test_xxxxxxxxxxxxxxxx",
    "webhook_secret": "your_webhook_secret"
}
```

**Test Credentials**: Get from https://korapay.com/dashboard

**Callback URL**: `https://yourdomain.com/payment/korapay/callback`  
**Webhook URL**: `https://yourdomain.com/payment/korapay/webhook`

#### Flutterwave Configuration
```php
{
    "public_key": "FLWPUBK_TEST-xxxxxxxxxxxxxxxx",
    "secret_key": "FLWSECK_TEST-xxxxxxxxxxxxxxxx",
    "encryption_key": "FLWSECK_TESTxxxxxxxx"
}
```

**Documentation**: https://developer.flutterwave.com  
**Callback URL**: `https://yourdomain.com/payment/flutterwave/callback`

#### Paystack Configuration
```php
{
    "public_key": "pk_test_xxxxxxxxxxxxxxxx",
    "secret_key": "sk_test_xxxxxxxxxxxxxxxx"
}
```

**Documentation**: https://paystack.com/docs  
**Callback URL**: `https://yourdomain.com/payment/paystack/callback`

#### Cryptomus Configuration
```php
{
    "merchant_id": "your_merchant_id",
    "api_key": "your_api_key",
    "currencies": ["BTC", "ETH", "USDT", "LTC"]
}
```

**Documentation**: https://doc.cryptomus.com  
**Callback URL**: `https://yourdomain.com/payment/cryptomus/callback`

### SMTP Email Configuration
```php
// config/app.php
'smtp_host' => 'smtp.gmail.com',
'smtp_port' => 587,
'smtp_username' => 'your-email@gmail.com',
'smtp_password' => 'your-app-password',
'smtp_encryption' => 'tls',
```

**Gmail App Password**: https://support.google.com/accounts/answer/185833

---

## 💳 Payment Gateways

### Implementation Overview

Each payment gateway follows this structure:
```
payments/
├── Korapay.php
├── Flutterwave.php
├── Paystack.php
├── Expresspay.php
├── Cryptomus.php
├── Heleket.php
└── Pesapal.php
```

### Payment Flow

```mermaid
sequenceDiagram
    User->>System: Initiate Payment
    System->>Gateway: Initialize Transaction
    Gateway-->>System: Return Checkout URL
    System->>User: Redirect to Gateway
    User->>Gateway: Complete Payment
    Gateway->>System: Webhook Notification
    System->>System: Verify Payment
    System->>Database: Update Balance
    System->>User: Confirmation + Credit
```

### Gateway Comparison

| Gateway | Countries | Currencies | Processing Time | Fees |
|---------|-----------|------------|-----------------|------|
| Korapay | Nigeria | NGN, USD | Instant | 1.5% |
| Flutterwave | Multi-country | Multiple | Instant | 1.4% |
| Paystack | Nigeria, Ghana, SA | NGN, GHS, ZAR | Instant | 1.5% |
| ExpressPay | Ghana | GHS, USD | Instant | 2.0% |
| Cryptomus | Global | Crypto | ~10 min | 1.0% |
| Heleket | Ethiopia | ETB | Instant | 2.5% |
| PesaPal | East Africa | KES, UGX, TZS | Instant | 2.0% |

---

## 🔄 Auto-Topup System

### How It Works

**Scenario**: User wants to place an order for $50 but only has $30 balance.

#### Step-by-Step Process:

**1. Order Initiation**
```php
// User places order
POST /orders/create
{
    "service_id": 123,
    "link": "https://instagram.com/username",
    "quantity": 1000
}

// System calculates: Order cost = $50
// User balance = $30
// Insufficient! Trigger auto-topup
```

**2. Auto-Topup Session Creation**
```sql
INSERT INTO auto_topup_sessions (
    user_id, session_token, service_id, link, quantity,
    order_amount, current_balance, required_amount, topup_amount,
    expires_at
) VALUES (
    1, 'token_abc123', 123, 'link', 1000,
    50.00, 30.00, 50.00, 20.00,
    NOW() + INTERVAL 30 MINUTE
);
```

**3. Payment Redirect**
```
User redirected to: /payment/select?session=token_abc123
Amount pre-filled: $20.00 (exact shortfall)
```

**4. Payment Completion**
```php
// After successful payment of $20:
// New balance: $30 + $20 = $50

// Automatic order placement:
1. Deduct $50 for order
2. Place order with provider
3. Remaining balance: $0

// If user paid $25 instead:
1. Deduct $50 for order
2. Place order
3. Credit remaining: $5
4. Final balance: $5
```

**5. Session Completion**
```sql
UPDATE auto_topup_sessions 
SET status = 'completed', 
    completed_at = NOW()
WHERE session_token = 'token_abc123';
```

### Database Schema

```sql
CREATE TABLE auto_topup_sessions (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT,
    session_token VARCHAR(64) UNIQUE,
    service_id INT,
    link TEXT,
    quantity INT,
    order_amount DECIMAL(15,4),
    current_balance DECIMAL(15,4),
    required_amount DECIMAL(15,4),
    topup_amount DECIMAL(15,4),
    transaction_id INT,
    status ENUM('pending', 'payment_processing', 'completed', 'expired', 'canceled'),
    expires_at DATETIME,
    completed_at DATETIME,
    created_at TIMESTAMP
);
```

### API Implementation

**OrderController.php**
```php
public function createOrder($data) {
    $servicePrice = $this->getServicePrice($data['service_id'], $data['quantity']);
    $userBalance = $this->getUserBalance($data['user_id']);
    
    if ($userBalance < $servicePrice) {
        // Insufficient balance - trigger auto-topup
        $session = $this->createAutoTopupSession([
            'user_id' => $data['user_id'],
            'service_id' => $data['service_id'],
            'link' => $data['link'],
            'quantity' => $data['quantity'],
            'order_amount' => $servicePrice,
            'current_balance' => $userBalance,
            'required_amount' => $servicePrice,
            'topup_amount' => $servicePrice - $userBalance
        ]);
        
        return [
            'success' => false,
            'insufficient_balance' => true,
            'topup_required' => true,
            'session_token' => $session['token'],
            'redirect_url' => '/payment/select?session=' . $session['token']
        ];
    }
    
    // Sufficient balance - proceed with order
    return $this->placeOrder($data);
}
```

---

## 🏗️ Architecture

### Directory Structure
```
advanced-smm-panel/
├── config/                 # Configuration files
│   ├── app.php
│   ├── database.php
│   └── payments.php
├── core/                   # Core framework
│   ├── Database.php
│   ├── Security.php
│   ├── Router.php
│   └── Controller.php
├── app/                    # Application layer
│   ├── controllers/
│   │   ├── AuthController.php
│   │   ├── OrderController.php
│   │   ├── PaymentController.php
│   │   └── UserController.php
│   ├── models/
│   │   ├── User.php
│   │   ├── Order.php
│   │   └── Transaction.php
│   └── views/
│       ├── auth/
│       ├── dashboard/
│       └── orders/
├── payments/               # Payment gateways
│   ├── Korapay.php
│   ├── Flutterwave.php
│   ├── Paystack.php
│   ├── Expresspay.php
│   ├── Cryptomus.php
│   ├── Heleket.php
│   └── Pesapal.php
├── public/                 # Public assets
│   ├── index.php          # Entry point
│   ├── index.html         # Landing page
│   ├── css/
│   └── js/
├── admin/                  # Admin panel
│   ├── controllers/
│   └── views/
├── cron/                   # Cron jobs
│   ├── process_orders.php
│   ├── sync_providers.php
│   └── cleanup.php
├── storage/                # Storage
│   └── logs/
└── database/               # Database
    └── schema.sql
```

### MVC Flow
```
Request → Router → Controller → Model → Database
                      ↓
                    View ← Response
```

### Database Design

**Core Tables**:
- `users` - User accounts
- `orders` - Order records
- `services` - Available services
- `transactions` - Financial transactions
- `payment_logs` - Payment audit trail
- `auto_topup_sessions` - Auto-topup tracking

**Relationships**:
```sql
users (1) ----→ (N) orders
orders (N) ----→ (1) services
orders (1) ----→ (1) transactions
users (1) ----→ (N) transactions
```

---

## 🔒 Security Features

### 1. Password Security
```php
// Argon2ID hashing
$hash = Security::hashPassword($password);
// Memory cost: 65536 KB
// Time cost: 4 iterations
// Parallelism: 3 threads
```

### 2. CSRF Protection
```php
// Generate token
$token = Security::generateCSRFToken();

// In forms
<?php echo Security::getCSRFField(); ?>

// Verify
if (!Security::verifyCSRFToken($_POST['csrf_token'])) {
    die('CSRF validation failed');
}
```

### 3. Rate Limiting
```php
// Check rate limit
if (!Security::checkRateLimit('login_' . $ip, 5, 15)) {
    die('Too many attempts. Try again in 15 minutes.');
}
```

### 4. SQL Injection Prevention
```php
// Always use prepared statements
$db->query("SELECT * FROM users WHERE email = :email")
   ->bind(':email', $email)
   ->getOne();
```

### 5. Session Security
- Session regeneration every 30 minutes
- Session timeout after 2 hours inactivity
- Session fingerprinting (User-Agent validation)
- Hijacking detection
- Secure cookie settings (HttpOnly, Secure, SameSite)

### 6. Security Headers
```php
X-Frame-Options: SAMEORIGIN
X-XSS-Protection: 1; mode=block
X-Content-Type-Options: nosniff
Strict-Transport-Security: max-age=31536000
Content-Security-Policy: default-src 'self'
```

### 7. Logging
```php
// Security events logged to: storage/logs/security.log
Security::logEvent('login_attempt', 'Failed login', $userId);
Security::logEvent('password_change', 'Password changed', $userId);
```

---

## 📡 API Documentation

### Base URL
```
https://yourdomain.com/api/v1
```

### Authentication
```http
Authorization: Bearer {api_key}
```

### Endpoints

#### **1. Get Balance**
```http
GET /api/v1/balance
Authorization: Bearer sk_xxxxxxxxxxxx

Response:
{
    "success": true,
    "balance": "125.50",
    "currency": "USD"
}
```

#### **2. Get Services**
```http
GET /api/v1/services
Authorization: Bearer sk_xxxxxxxxxxxx

Response:
{
    "success": true,
    "services": [
        {
            "id": 1,
            "name": "Instagram Followers",
            "price_per_1000": "5.00",
            "min": 100,
            "max": 100000
        }
    ]
}
```

#### **3. Place Order**
```http
POST /api/v1/orders
Authorization: Bearer sk_xxxxxxxxxxxx
Content-Type: application/json

{
    "service_id": 1,
    "link": "https://instagram.com/username",
    "quantity": 1000
}

Response:
{
    "success": true,
    "order_id": 12345,
    "charge": "5.00",
    "status": "pending"
}
```

#### **4. Check Order Status**
```http
GET /api/v1/orders/{order_id}
Authorization: Bearer sk_xxxxxxxxxxxx

Response:
{
    "success": true,
    "order": {
        "id": 12345,
        "status": "completed",
        "start_count": 1000,
        "remains": 0,
        "created_at": "2026-02-17 10:30:00"
    }
}
```

#### **5. Add Funds**
```http
POST /api/v1/payments/initialize
Authorization: Bearer sk_xxxxxxxxxxxx

{
    "method": "korapay",
    "amount": 50.00,
    "currency": "NGN"
}

Response:
{
    "success": true,
    "checkout_url": "https://checkout.korapay.com/xxxxx",
    "reference": "KRP_1234567890_abcdef"
}
```

### Rate Limiting
- 60 requests per minute per API key
- 429 status code when exceeded
- Reset header: `X-RateLimit-Reset`

### Error Codes
```json
{
    "success": false,
    "error": {
        "code": "INSUFFICIENT_BALANCE",
        "message": "Insufficient balance to complete order",
        "details": {
            "required": "50.00",
            "available": "30.00"
        }
    }
}
```

---

## 🚀 Deployment

### Production Checklist

**1. Security**
- [ ] Change default admin password
- [ ] Generate new encryption key
- [ ] Enable HTTPS/SSL
- [ ] Set `app_env` to 'production'
- [ ] Set `app_debug` to false
- [ ] Configure firewall
- [ ] Set up fail2ban

**2. Database**
- [ ] Create production database
- [ ] Import schema
- [ ] Create database user with limited privileges
- [ ] Enable query caching
- [ ] Set up automated backups

**3. Payment Gateways**
- [ ] Switch to live API keys
- [ ] Configure webhooks
- [ ] Test each gateway with small amounts
- [ ] Set up payment monitoring

**4. Server**
- [ ] PHP 8.0+ installed
- [ ] MySQL 8.0+ installed
- [ ] Enable OPcache
- [ ] Set memory_limit to 256M
- [ ] Enable Gzip compression
- [ ] Configure SSL certificate

**5. Cron Jobs**
```bash
# Edit crontab
crontab -e

# Add these lines
* * * * * php /path/to/panel/cron/process_orders.php >> /dev/null 2>&1
*/5 * * * * php /path/to/panel/cron/sync_providers.php >> /dev/null 2>&1
0 2 * * * php /path/to/panel/cron/cleanup.php >> /dev/null 2>&1
```

**6. Monitoring**
- [ ] Set up error monitoring (Sentry, Bugsnag)
- [ ] Configure uptime monitoring
- [ ] Set up payment alerts
- [ ] Monitor security logs
- [ ] Track API usage

### Nginx Configuration
```nginx
server {
    listen 443 ssl http2;
    server_name yourdomain.com;
    
    ssl_certificate /path/to/cert.pem;
    ssl_certificate_key /path/to/key.pem;
    
    root /path/to/panel/public;
    index index.php index.html;
    
    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }
    
    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.2-fpm.sock;
        fastcgi_index index.php;
        include fastcgi_params;
    }
    
    location ~ /\.ht {
        deny all;
    }
}
```

### Apache Configuration
```apache
<VirtualHost *:443>
    ServerName yourdomain.com
    DocumentRoot /path/to/panel/public
    
    SSLEngine on
    SSLCertificateFile /path/to/cert.pem
    SSLCertificateKeyFile /path/to/key.pem
    
    <Directory /path/to/panel/public>
        AllowOverride All
        Require all granted
    </Directory>
    
    ErrorLog ${APACHE_LOG_DIR}/panel_error.log
    CustomLog ${APACHE_LOG_DIR}/panel_access.log combined
</VirtualHost>
```

---

## 🔧 Troubleshooting

### Common Issues

**1. Database Connection Error**
```
Solution:
- Check config/database.php credentials
- Verify MySQL is running: systemctl status mysql
- Check user privileges: GRANT ALL ON smm_panel.* TO 'user'@'localhost';
```

**2. Payment Gateway Errors**
```
Solution:
- Verify API keys are correct (live vs test)
- Check webhook URLs are publicly accessible
- Review payment_logs table for details
- Test with curl commands
```

**3. Auto-Topup Not Working**
```
Solution:
- Check auto_topup_enabled setting
- Verify session hasn't expired (30 min default)
- Check payment webhook is receiving notifications
- Review auto_topup_sessions table
```

**4. Orders Stuck in Pending**
```
Solution:
- Verify cron jobs are running: crontab -l
- Check provider API credentials
- Review order logs: SELECT * FROM orders WHERE status = 'pending';
- Manually trigger: php cron/process_orders.php
```

**5. Email Not Sending**
```
Solution:
- Verify SMTP credentials in config/app.php
- Check firewall allows port 587/465
- Test with: php -r "mail('test@example.com', 'Test', 'Body');"
- Review mail logs: /var/log/mail.log
```

### Debug Mode

**Enable temporarily for troubleshooting:**
```php
// config/app.php
'app_env' => 'development',
'app_debug' => true,

// Then disable after fixing!
```

### Logs Location
```
storage/logs/security.log    - Security events
storage/logs/errors.log       - PHP errors
storage/logs/payments.log     - Payment activities
storage/logs/api.log          - API requests
```

---

## 📊 Performance Optimization

### 1. Database Optimization
```sql
-- Add indexes
CREATE INDEX idx_orders_status ON orders(status);
CREATE INDEX idx_transactions_user ON transactions(user_id);
CREATE INDEX idx_payment_logs_method ON payment_logs(payment_method);

-- Optimize tables monthly
OPTIMIZE TABLE orders, transactions, payment_logs;
```

### 2. PHP Optimization
```ini
; php.ini
opcache.enable=1
opcache.memory_consumption=256
opcache.max_accelerated_files=10000
opcache.revalidate_freq=0
```

### 3. Caching
```php
// Implement Redis/Memcached for sessions
ini_set('session.save_handler', 'redis');
ini_set('session.save_path', 'tcp://127.0.0.1:6379');
```

---

## 📞 Support

### Resources
- Documentation: https://docs.yourdomain.com
- API Reference: https://api.yourdomain.com/docs
- Community: https://community.yourdomain.com
- Email: support@yourdomain.com

### System Requirements
- PHP 8.0+
- MySQL 8.0+
- SSL Certificate
- Cron access
- Minimum 2GB RAM
- 10GB disk space

---

## 📝 License

Copyright © 2026 Advanced SMM Panel. All rights reserved.

---

## 🎉 Congratulations!

Your advanced SMM panel is now ready. Key features:
✅ Modern, animated landing page  
✅ Revolutionary auto-topup system  
✅ 8 payment gateways integrated  
✅ Bank-grade security  
✅ Complete API  
✅ Production-ready code  

**Next Steps:**
1. Configure payment gateways
2. Add services and categories
3. Test auto-topup flow
4. Deploy to production
5. Monitor and scale

**Need Help?**  
Review this documentation, check logs, and reach out to support.

Happy selling! 🚀
